using System.Collections.Generic;
using UnityEngine;
using UnityEngine.SceneManagement;
using UnityEngine.XR.ARFoundation;
using UnityEngine.XR.ARSubsystems;
using TMPro;

[RequireComponent(typeof(ARRaycastManager))]
public class MultiMeasureTool : MonoBehaviour
{
    [Header("Prefabs")]
    public GameObject reticlePrefab;
    public GameObject pointPrefab;
    public LineRenderer lineRendererPrefab;
    public TextMeshPro distanceTextPrefab;

    private ARRaycastManager raycastManager;
    private GameObject reticle;
    private static List<ARRaycastHit> hits = new List<ARRaycastHit>();

    private List<Measurement> measurements = new List<Measurement>();

    private GameObject tempPoint;
    private LineRenderer previewLine;
    private TextMeshPro previewText;

    public void LoadHome()
    {
        SceneManager.LoadScene("Home");
        // Debug.Log("Home");
    }

    [System.Serializable]
    private class Measurement
    {
        public GameObject point1;
        public GameObject point2;
        public LineRenderer line;
        public TextMeshPro distanceText;
    }

    void Start()
    {
        raycastManager = GetComponent<ARRaycastManager>();

        reticle = Instantiate(reticlePrefab);
        reticle.SetActive(false);

        previewLine = Instantiate(lineRendererPrefab);
        previewLine.positionCount = 2;
        previewLine.enabled = false;

        previewText = Instantiate(distanceTextPrefab, Vector3.zero, Quaternion.identity);
        previewText.gameObject.SetActive(false);
    }


    void Update()
    {
        UpdateReticle();

        if (Input.touchCount > 0 && Input.GetTouch(0).phase == TouchPhase.Began)
        {
            if (reticle.activeSelf)
            {
                PlacePoint();
            }
        }

        if (tempPoint != null && reticle.activeSelf)
        {
            DrawPreviewLine(tempPoint.transform.position, reticle.transform.position);
            UpdateDistanceText(previewText, tempPoint.transform.position, reticle.transform.position);
        }
    }

    void UpdateReticle()
    {
        Vector2 screenCenter = new Vector2(Screen.width / 2, Screen.height / 2);

        if (raycastManager.Raycast(screenCenter, hits, TrackableType.PlaneWithinPolygon))
        {
            Pose hitPose = hits[0].pose;
            reticle.SetActive(true);
            reticle.transform.position = hitPose.position;
            reticle.transform.rotation = hitPose.rotation;
        }
        else
        {
            reticle.SetActive(false);
        }
    }

    void PlacePoint()
    {
        if (tempPoint == null)
        {
            tempPoint = Instantiate(pointPrefab, reticle.transform.position, Quaternion.identity);

            previewLine.enabled = true;
            previewText.gameObject.SetActive(true);
        }
        else
        {
            GameObject point2 = Instantiate(pointPrefab, reticle.transform.position, Quaternion.identity);

            LineRenderer newLine = Instantiate(lineRendererPrefab);
            newLine.positionCount = 2;
            newLine.SetPosition(0, tempPoint.transform.position);
            newLine.SetPosition(1, point2.transform.position);

            TextMeshPro newText = Instantiate(distanceTextPrefab, Vector3.zero, Quaternion.identity);
            UpdateDistanceText(newText, tempPoint.transform.position, point2.transform.position);

            Measurement m = new Measurement
            {
                point1 = tempPoint,
                point2 = point2,
                line = newLine,
                distanceText = newText
            };
            measurements.Add(m);

            previewLine.enabled = false;
            previewText.gameObject.SetActive(false);

            tempPoint = null;
        }
    }

    void DrawPreviewLine(Vector3 start, Vector3 end)
    {
        previewLine.enabled = true;
        previewLine.SetPosition(0, start);
        previewLine.SetPosition(1, end);
    }

    void UpdateDistanceText(TextMeshPro text, Vector3 start, Vector3 end)
    {
        float distance = Vector3.Distance(start, end) * 100f; // meter → cm
        text.text = distance.ToString("F1") + " cm";

        Vector3 midPoint = (start + end) / 2f;
        text.transform.position = midPoint;

        Camera cam = Camera.main;
        if (cam != null)
        {
            text.transform.rotation =
                Quaternion.LookRotation(text.transform.position - cam.transform.position);
        }
    }
}
